import React, { useState } from 'react';
import { Check, Loader2, X, Facebook } from 'lucide-react';

interface FacebookTaskModalProps {
  isOpen: boolean;
  onComplete: () => void;
  onClose: () => void;
}

export const FacebookTaskModal: React.FC<FacebookTaskModalProps> = ({
  isOpen,
  onComplete,
  onClose
}) => {
  const [isFollowing, setIsFollowing] = useState(false);
  const [isCompleted, setIsCompleted] = useState(false);

  const handleFollowClick = () => {
    if (isCompleted || isFollowing) {
      return;
    }

    // Set loading state
    setIsFollowing(true);

    // Open Facebook page
    window.open('https://facebook.com/dexapkcom', '_blank', 'noopener,noreferrer');

    // Simulate verification process (6 seconds)
    setTimeout(() => {
      setIsFollowing(false);
      setIsCompleted(true);
    }, 6000);
  };

  const handleProceed = () => {
    if (isCompleted) {
      onComplete();
    }
  };

  if (!isOpen) return null;

  return (
    <div 
      className="fixed inset-0 bg-black bg-opacity-60 flex items-center justify-center p-4 z-50"
      role="dialog"
      aria-modal="true"
      aria-labelledby="facebook-task-title"
    >
      <div className="bg-white rounded-2xl shadow-2xl max-w-md w-full p-6 relative max-h-[90vh] overflow-y-auto">
        {/* Close button */}
        <button
          onClick={onClose}
          className="absolute top-4 right-4 p-2 text-gray-400 hover:text-gray-600 rounded-full z-10"
          aria-label="Close"
        >
          <X className="w-5 h-5" />
        </button>

        {/* Header */}
        <div className="text-center mb-6">
          <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4 relative">
            <Facebook className="w-8 h-8 text-blue-600" />
          </div>
          <h2 id="facebook-task-title" className="text-2xl font-bold text-gray-900 mb-2">
            Complete Process Task!
          </h2>
          <p className="text-gray-600">
            Follow our Facebook Page to continue
          </p>
        </div>

        {/* Task Banner */}
        <div className="bg-blue-50 rounded-xl p-4 mb-6 border border-blue-200">
          <div className="flex items-center space-x-3">
            <Facebook className="w-6 h-6 text-blue-600 flex-shrink-0" />
            <div>
              <h3 className="font-semibold text-blue-900">Follow our Facebook Page to continue</h3>
              <p className="text-sm text-blue-700 mt-1">
                Stay updated with our latest offers and services
              </p>
            </div>
          </div>
        </div>

        {/* Follow Facebook Button */}
        <button
          onClick={handleFollowClick}
          disabled={isFollowing || isCompleted}
          className={`
            w-full mb-6 px-6 py-4 rounded-xl font-semibold text-lg
            transition-all duration-200 min-h-[64px] flex items-center justify-center
            focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
            ${isCompleted
              ? 'bg-green-100 text-green-800 cursor-not-allowed'
              : isFollowing
              ? 'bg-blue-500 text-white cursor-not-allowed'
              : 'bg-blue-600 text-white hover:bg-blue-700'
            }
          `}
        >
          {isFollowing ? (
            <>
              <Loader2 className="w-5 h-5 mr-2 animate-spin" />
              Verifying...
            </>
          ) : isCompleted ? (
            <>
              <Check className="w-5 h-5 mr-2" />
              Followed Successfully!
            </>
          ) : (
            <>
              <Facebook className="w-5 h-5 mr-2" />
              Follow Facebook
            </>
          )}
        </button>

        {/* Instructions */}
        {!isCompleted && (
          <div className="bg-gray-50 rounded-xl p-4 mb-6">
            <h3 className="font-semibold text-gray-900 mb-2">How it works:</h3>
            <ol className="text-sm text-gray-700 space-y-1">
              <li>1. Click "Follow Facebook" button</li>
              <li>2. Facebook page will open in a new tab</li>
              <li>3. Click the "Follow" or "Like" button on our page</li>
              <li>4. Return here and wait for verification</li>
            </ol>
          </div>
        )}

        {/* Proceed Button */}
        <button
          onClick={handleProceed}
          disabled={!isCompleted}
          className={`
            w-full px-6 py-4 rounded-xl font-semibold text-lg
            transition-all duration-200 min-h-[56px]
            focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
            ${isCompleted
              ? 'bg-blue-600 text-white hover:bg-blue-700'
              : 'bg-gray-300 text-gray-500 cursor-not-allowed'
            }
          `}
        >
          {isCompleted ? 'Proceed' : 'Complete Facebook Follow First'}
        </button>

        {/* Footer Note */}
        <p className="text-xs text-gray-500 text-center mt-4">
          Following our page helps us provide better services and updates
        </p>
      </div>
    </div>
  );
};