import React, { useState, useEffect } from 'react';

interface PhoneInputProps {
  countryCode: string;
  value: string;
  onChange: (value: string) => void;
  onValidationChange: (isValid: boolean) => void;
  error?: string;
}

export const PhoneInput: React.FC<PhoneInputProps> = ({
  countryCode,
  value,
  onChange,
  onValidationChange,
  error
}) => {
  const [inputValue, setInputValue] = useState(value);
  const [isValid, setIsValid] = useState(false);

  const validatePhoneNumber = (number: string) => {
    // For India, expect 10 digits
    if (countryCode === '+91') {
      const digitsOnly = number.replace(/\D/g, '');
      return digitsOnly.length === 10;
    }
    // Basic validation for other countries (at least 7 digits)
    const digitsOnly = number.replace(/\D/g, '');
    return digitsOnly.length >= 7;
  };

  useEffect(() => {
    const valid = validatePhoneNumber(inputValue);
    setIsValid(valid);
    onValidationChange(valid);
  }, [inputValue, countryCode, onValidationChange]);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    let value = e.target.value;
    
    // Remove any non-digits
    value = value.replace(/\D/g, '');
    
    // Limit length based on country
    if (countryCode === '+91' && value.length > 10) {
      value = value.substring(0, 10);
    } else if (value.length > 15) {
      value = value.substring(0, 15);
    }
    
    setInputValue(value);
    onChange(value);
  };

  const handlePaste = (e: React.ClipboardEvent<HTMLInputElement>) => {
    e.preventDefault();
    const pastedText = e.clipboardData.getData('text/plain');
    const digitsOnly = pastedText.replace(/\D/g, '');
    
    let limitedDigits = digitsOnly;
    if (countryCode === '+91' && digitsOnly.length > 10) {
      limitedDigits = digitsOnly.substring(0, 10);
    } else if (digitsOnly.length > 15) {
      limitedDigits = digitsOnly.substring(0, 15);
    }
    
    setInputValue(limitedDigits);
    onChange(limitedDigits);
  };

  const inputId = 'phone-input';
  const errorId = `${inputId}-error`;
  const helpId = `${inputId}-help`;

  return (
    <div className="space-y-2">
      <label htmlFor={inputId} className="block text-sm font-medium text-gray-700">
        Phone Number
      </label>
      <div className="relative">
        <div className="flex">
          <div className="
            flex items-center px-4 py-3 bg-gray-50 border border-gray-300 border-r-0 
            rounded-l-xl text-sm font-medium text-gray-700
          ">
            {countryCode}
          </div>
          <input
            id={inputId}
            type="tel"
            value={inputValue}
            onChange={handleInputChange}
            onPaste={handlePaste}
            className={`
              flex-1 px-4 py-3 border rounded-r-xl text-lg
              focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500
              ${error || !isValid && inputValue ? 'border-red-300 text-red-900' : 'border-gray-300'}
            `}
            placeholder="Enter phone number"
            aria-invalid={error || (!isValid && inputValue) ? 'true' : 'false'}
            aria-describedby={`${helpId} ${error ? errorId : ''}`}
          />
        </div>
      </div>
      
      <div id={helpId} className="text-sm text-gray-500">
        {countryCode === '+91' 
          ? 'Enter your 10-digit mobile number'
          : 'Enter your mobile number'
        }
      </div>
      
      {error && (
        <div id={errorId} className="text-sm text-red-600" role="alert">
          {error}
        </div>
      )}
      
      {!isValid && inputValue && !error && (
        <div className="text-sm text-red-600" role="alert">
          {countryCode === '+91' 
            ? 'Please enter a valid 10-digit mobile number'
            : 'Please enter a valid mobile number'
          }
        </div>
      )}
    </div>
  );
};