import React from 'react';
import { Plan } from '../types';

interface PlanCardProps {
  plan: Plan;
  onClick: () => void;
}

export const PlanCard: React.FC<PlanCardProps> = ({ plan, onClick }) => {
  const formatPrice = (price: number, currency: string) => {
    const symbol = currency === 'INR' ? '₹' : '$';
    return `${symbol}${price.toLocaleString()}`;
  };

  const formatValidity = (days: number) => {
    if (days >= 365) {
      return `${Math.round(days / 365)} year${days >= 730 ? 's' : ''}`;
    }
    if (days >= 30) {
      return `${Math.round(days / 30)} month${days >= 60 ? 's' : ''}`;
    }
    return `${days} days`;
  };

  return (
    <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-6 hover:shadow-md transition-all duration-200">
      <div className="flex justify-between items-start mb-4">
        <div className="flex-1">
          <div className="text-4xl font-bold text-gray-900 mb-2">
            {formatPrice(plan.price, plan.currency)}
          </div>
          <div className="flex flex-wrap items-center gap-4 text-sm text-gray-600 mb-3">
            <span className="flex items-center">
              <span className="font-medium">Validity:</span>
              <span className="ml-1">{formatValidity(plan.validityDays)}</span>
            </span>
            <span className="flex items-center">
              <span className="font-medium">Data:</span>
              <span className="ml-1">{plan.data}</span>
            </span>
          </div>
          {plan.badges.length > 0 && (
            <div className="flex flex-wrap gap-2 mb-4">
              {plan.badges.map((badge) => (
                <span
                  key={badge}
                  className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800"
                >
                  {badge}
                </span>
              ))}
            </div>
          )}
        </div>
        <button
          onClick={onClick}
          className="
            relative px-6 py-3 bg-blue-600 text-white font-semibold rounded-xl
            hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
            transition-colors duration-200 min-h-[48px]
          "
        >
          <div className="flex flex-col items-center">
            <span className="text-xs font-bold text-blue-200 leading-none">FREE</span>
            <span className="text-sm font-semibold leading-none">Buy</span>
          </div>
        </button>
      </div>
    </div>
  );
};