import React, { useEffect, useState } from 'react';
import { Check, X } from 'lucide-react';

interface ProcessingModalProps {
  isOpen: boolean;
  onSuccess: (orderRef: string) => void;
  onError: () => void;
  providerName: string;
  providerLogo: string;
  phoneNumber: string;
  amount: number;
  currency: string;
}

type ProcessingState = 'validating' | 'connecting' | 'processing' | 'finalizing' | 'success' | 'error';

interface StateConfig {
  en: string;
  hi: string;
  duration: [number, number]; // min, max in ms
}

const stateConfigs: Record<ProcessingState, StateConfig> = {
  validating: {
    en: 'Validating number...',
    hi: 'मोबाइल नंबर सत्यापित किया जा रहा है...',
    duration: [600, 900]
  },
  connecting: {
    en: 'Connecting to {provider}...',
    hi: '{provider} से कनेक्ट किया जा रहा है...',
    duration: [1000, 1500]
  },
  processing: {
    en: 'Securing transaction...',
    hi: 'लेनदेन सुरक्षित किया जा रहा है...',
    duration: [900, 1200]
  },
  finalizing: {
    en: 'Finalizing...',
    hi: 'अंतिम चरण...',
    duration: [400, 600]
  },
  success: {
    en: 'Transaction completed!',
    hi: 'लेनदेन पूरा हो गया!',
    duration: [500, 500]
  },
  error: {
    en: 'We couldn\'t complete the recharge. Please try again.',
    hi: 'रिचार्ज पूरा नहीं हो सका। कृपया दोबारा प्रयास करें।',
    duration: [0, 0]
  }
};

export const ProcessingModal: React.FC<ProcessingModalProps> = ({
  isOpen,
  onSuccess,
  onError,
  providerName,
  providerLogo,
  phoneNumber,
  amount,
  currency
}) => {
  const [currentState, setCurrentState] = useState<ProcessingState>('validating');
  const [progress, setProgress] = useState(0);
  const [isComplete, setIsComplete] = useState(false);

  const generateOrderRef = () => {
    return 'REF' + Math.random().toString(36).substr(2, 9).toUpperCase();
  };

  const getRandomDuration = (state: ProcessingState) => {
    const [min, max] = stateConfigs[state].duration;
    return Math.floor(Math.random() * (max - min + 1)) + min;
  };

  const getCurrentText = (state: ProcessingState, lang: 'en' | 'hi') => {
    const config = stateConfigs[state];
    return config[lang].replace('{provider}', providerName);
  };

  useEffect(() => {
    if (!isOpen) {
      setCurrentState('validating');
      setProgress(0);
      setIsComplete(false);
      return;
    }

    let timeoutId: NodeJS.Timeout;
    let progressInterval: NodeJS.Timeout;

    const processStates = async () => {
      const states: ProcessingState[] = ['validating', 'connecting', 'processing', 'finalizing'];
      
      // Simulate 5% failure rate
      const shouldFail = Math.random() < 0.05;
      
      for (let i = 0; i < states.length; i++) {
        const state = states[i];
        setCurrentState(state);
        
        const duration = getRandomDuration(state);
        const startProgress = (i / states.length) * 100;
        const endProgress = ((i + 1) / states.length) * 100;
        
        // Animate progress bar
        let currentProgress = startProgress;
        const progressStep = (endProgress - startProgress) / (duration / 50);
        
        progressInterval = setInterval(() => {
          currentProgress = Math.min(currentProgress + progressStep, endProgress);
          setProgress(currentProgress);
        }, 50);
        
        await new Promise(resolve => {
          timeoutId = setTimeout(resolve, duration);
        });
        
        clearInterval(progressInterval);
        
        // Check for failure at processing stage
        if (shouldFail && state === 'processing') {
          setCurrentState('error');
          setProgress(0);
          return;
        }
      }
      
      // Success
      setProgress(100);
      setCurrentState('success');
      setIsComplete(true);
      
      // Auto-navigate after showing success briefly
      setTimeout(() => {
        const orderRef = generateOrderRef();
        onSuccess(orderRef);
      }, 800);
    };

    processStates();

    // Timeout fallback (8 seconds)
    const fallbackTimeout = setTimeout(() => {
      setCurrentState('error');
      setProgress(0);
    }, 8000);

    return () => {
      clearTimeout(timeoutId);
      clearTimeout(fallbackTimeout);
      clearInterval(progressInterval);
    };
  }, [isOpen, onSuccess, providerName]);

  const handleRetry = () => {
    setCurrentState('validating');
    setProgress(0);
    setIsComplete(false);
  };

  const handleClose = () => {
    onError();
  };

  if (!isOpen) return null;

  return (
    <div 
      className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50"
      role="dialog"
      aria-modal="true"
      aria-labelledby="processing-title"
    >
      <div className="bg-white rounded-2xl shadow-xl max-w-md w-full p-8 text-center">
        <h2 id="processing-title" className="text-xl font-semibold text-gray-900 mb-6">
          {currentState === 'error' ? 'Transaction Failed' : 'Processing Recharge'}
        </h2>

        {currentState !== 'error' && (
          <>
            {/* Progress Bar */}
            <div className="w-full bg-gray-200 rounded-full h-2 mb-6">
              <div 
                className="bg-blue-600 h-2 rounded-full transition-all duration-300 ease-out"
                style={{ width: `${progress}%` }}
              />
            </div>

            {/* Provider Logo (shown during connecting state) */}
            {currentState === 'connecting' && (
              <div className="flex justify-center mb-6">
                <div className="w-16 h-16 rounded-full overflow-hidden animate-pulse">
                  <img
                    src={providerLogo}
                    alt={`${providerName} logo`}
                    className="w-full h-full object-cover"
                    onError={(e) => {
                      (e.target as HTMLImageElement).style.display = 'none';
                    }}
                  />
                </div>
              </div>
            )}

            {/* Success Checkmark */}
            {currentState === 'success' && (
              <div className="flex justify-center mb-6">
                <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center">
                  <Check className="w-8 h-8 text-green-600" />
                </div>
              </div>
            )}
          </>
        )}

        {/* Error Icon */}
        {currentState === 'error' && (
          <div className="flex justify-center mb-6">
            <div className="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center">
              <X className="w-8 h-8 text-red-600" />
            </div>
          </div>
        )}

        {/* Status Text with Live Region */}
        <div 
          className="space-y-2 mb-6"
          aria-live="polite" 
          role="status"
        >
          <p className="text-gray-900 font-medium">
            {getCurrentText(currentState, 'en')}
          </p>
          <p className="text-gray-600 text-sm" lang="hi">
            {getCurrentText(currentState, 'hi')}
          </p>
        </div>

        {/* Error Actions */}
        {currentState === 'error' && (
          <div className="flex flex-col sm:flex-row gap-3">
            <button
              onClick={handleRetry}
              className="
                flex-1 px-6 py-3 bg-blue-600 text-white font-semibold rounded-xl
                hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
                transition-colors duration-200
              "
            >
              Retry
            </button>
            <button
              onClick={handleClose}
              className="
                flex-1 px-6 py-3 bg-gray-200 text-gray-800 font-semibold rounded-xl
                hover:bg-gray-300 focus:outline-none focus:ring-2 focus:ring-gray-500 focus:ring-offset-2
                transition-colors duration-200
              "
            >
              Back to Plans
            </button>
          </div>
        )}
      </div>
    </div>
  );
};