import React, { useState } from 'react';
import { Check, Loader2, X, MessageCircle } from 'lucide-react';

interface QuestionnaireModalProps {
  isOpen: boolean;
  onComplete: () => void;
  onClose: () => void;
}

type QuestionStep = 1 | 2 | 3;
type Answer = 'yes' | 'no' | null;
type VerificationState = 'idle' | 'waiting' | 'ready' | 'verifying' | 'complete';

export const QuestionnaireModal: React.FC<QuestionnaireModalProps> = ({
  isOpen,
  onComplete,
  onClose
}) => {
  const [currentStep, setCurrentStep] = useState<QuestionStep>(1);
  const [answers, setAnswers] = useState<Record<number, Answer>>({});
  const [verificationState, setVerificationState] = useState<VerificationState>('idle');
  const [pendingTelegramUrl, setPendingTelegramUrl] = useState<string>('');

  const questions = [
    {
      id: 1,
      text: '1. क्या आप अपना Mobile नंबर फ्री में Recharge करना चाहते हैं?',
      options: ['हाँ', 'नहीं']
    },
    {
      id: 2,
      text: '2. क्या आप Color Prediction गेम के बारे में जानते हैं?',
      options: ['हाँ', 'नहीं']
    },
    {
      id: 3,
      text: '3. क्या आप गेम खेलकर पैसे कमाना चाहते हैं?',
      options: ['हाँ', 'नहीं']
    }
  ];

  const handleAnswer = (questionId: number, answer: 'yes' | 'no') => {
    setAnswers(prev => ({ ...prev, [questionId]: answer }));

    if (questionId === 1) {
      // Always proceed to question 2
      setCurrentStep(2);
    } else if (questionId === 2) {
      if (answer === 'yes') {
        // Show verification screen for Color Prediction
        setPendingTelegramUrl('https://t.me/+89-0JDD06uJjNTk1');
        setVerificationState('waiting');
        
        // 3-second timer before enabling the button
        setTimeout(() => {
          setVerificationState('ready');
        }, 3000);
      } else {
        // Proceed to question 3
        setCurrentStep(3);
      }
    } else if (questionId === 3) {
      if (answer === 'yes') {
        // Show verification screen for Gaming
        setPendingTelegramUrl('https://t.me/+n9qQ5GadgltiMjI1');
        setVerificationState('waiting');
        
        // 3-second timer before enabling the button
        setTimeout(() => {
          setVerificationState('ready');
        }, 3000);
      } else {
        // Complete questionnaire
        onComplete();
      }
    }
  };

  const handleTelegramJoin = () => {
    if (verificationState !== 'ready') return;
    
    setVerificationState('verifying');

    // Open Telegram channel
    window.open(pendingTelegramUrl, '_blank', 'noopener,noreferrer');

    // 7-second verification process
    setTimeout(() => {
      setVerificationState('complete');

      // Auto-proceed after verification
      setTimeout(() => {
        if (currentStep === 2) {
          setCurrentStep(3);
          setVerificationState('idle');
          setPendingTelegramUrl('');
        } else if (currentStep === 3) {
          onComplete();
        }
      }, 1000);
    }, 7000);
  };

  const currentQuestion = questions.find(q => q.id === currentStep);
  const isShowingVerification = verificationState !== 'idle';

  if (!isOpen) return null;

  return (
    <div 
      className="fixed inset-0 bg-black bg-opacity-60 flex items-center justify-center p-4 z-50"
      role="dialog"
      aria-modal="true"
      aria-labelledby="questionnaire-title"
    >
      <div className="bg-white rounded-2xl shadow-2xl max-w-md w-full p-6 relative max-h-[90vh] overflow-y-auto">
        {/* Close button - disabled during verification */}
        <button
          onClick={onClose}
          disabled={isShowingVerification}
          className={`
            absolute top-4 right-4 p-2 rounded-full z-10
            ${isShowingVerification 
              ? 'text-gray-300 cursor-not-allowed' 
              : 'text-gray-400 hover:text-gray-600'
            }
          `}
          aria-label="Close"
        >
          <X className="w-5 h-5" />
        </button>

        {/* Header */}
        <div className="text-center mb-6">
          <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <span className="text-2xl">❓</span>
          </div>
          <h2 id="questionnaire-title" className="text-2xl font-bold text-gray-900 mb-2">
            Quick Questions
          </h2>
          <p className="text-gray-600">
            Please answer these questions to continue
          </p>
        </div>

        {/* Progress Indicator */}
        {!isShowingVerification && (
          <div className="flex justify-center mb-6">
            <div className="flex space-x-2">
              {[1, 2, 3].map((step) => (
                <div
                  key={step}
                  className={`
                    w-3 h-3 rounded-full transition-all duration-300
                    ${step === currentStep 
                      ? 'bg-blue-600' 
                      : step < currentStep 
                      ? 'bg-green-500' 
                      : 'bg-gray-300'
                    }
                  `}
                />
              ))}
            </div>
            <span className="ml-3 text-sm text-gray-600">
              {currentStep} of 3
            </span>
          </div>
        )}

        {/* Telegram Verification Screen */}
        {isShowingVerification && (
          <div className="text-center mb-6">
            <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <MessageCircle className="w-8 h-8 text-blue-600" />
            </div>
            
            {verificationState === 'waiting' && (
              <>
                <div className="flex items-center justify-center space-x-2 mb-4">
                  <Loader2 className="w-5 h-5 animate-spin text-blue-600" />
                  <span className="text-blue-600 font-medium">Preparing Telegram join...</span>
                </div>
                <div className="bg-blue-50 rounded-xl p-4">
                  <p className="text-sm text-blue-800">
                    Please wait while we prepare your Telegram channel access.
                  </p>
                </div>
              </>
            )}
            
            {verificationState === 'ready' && (
              <>
                <h3 className="text-lg font-semibold text-gray-900 mb-4">
                  Join Telegram Channel
                </h3>
                <div className="bg-blue-50 rounded-xl p-4 mb-6">
                  <p className="text-sm text-blue-800">
                    Click the button below to join our Telegram channel. After joining, wait for verification to complete.
                  </p>
                </div>
                <button
                  onClick={handleTelegramJoin}
                  className="
                    w-full px-6 py-4 rounded-xl font-semibold text-lg
                    bg-blue-600 text-white hover:bg-blue-700
                    focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
                    transition-all duration-200 min-h-[56px] flex items-center justify-center
                  "
                >
                  <MessageCircle className="w-5 h-5 mr-2" />
                  Join Telegram Channel
                </button>
              </>
            )}
            
            {verificationState === 'verifying' && (
              <>
                <div className="flex items-center justify-center space-x-2 mb-4">
                  <Loader2 className="w-5 h-5 animate-spin text-blue-600" />
                  <span className="text-blue-600 font-medium">Verifying Telegram join...</span>
                </div>
                <div className="bg-blue-50 rounded-xl p-4">
                  <p className="text-sm text-blue-800">
                    Please join the Telegram channel that opened in a new tab, then wait for verification to complete.
                  </p>
                </div>
              </>
            )}
            
            {verificationState === 'complete' && (
              <>
                <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Check className="w-8 h-8 text-green-600" />
                </div>
                <p className="text-green-600 font-medium">Verification Complete!</p>
              </>
            )}
          </div>
        )}

        {/* Question Content */}
        {!isShowingVerification && currentQuestion && (
          <div className="mb-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-6 text-center leading-relaxed">
              {currentQuestion.text}
            </h3>

            <div className="space-y-3">
              {currentQuestion.options.map((option, index) => {
                const answerValue = index === 0 ? 'yes' : 'no';
                
                return (
                  <button
                    key={option}
                    onClick={() => handleAnswer(currentQuestion.id, answerValue)}
                    className="
                      w-full px-6 py-4 rounded-xl font-semibold text-lg
                      bg-blue-600 text-white hover:bg-blue-700
                      focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
                      transition-all duration-200 min-h-[56px]
                      active:scale-95
                    "
                  >
                    {option}
                  </button>
                );
              })}
            </div>
          </div>
        )}

        {/* Instructions for verification states */}
        {verificationState === 'waiting' && (
          <div className="bg-gray-50 rounded-xl p-4 mb-6">
            <h4 className="font-semibold text-gray-900 mb-2">Please Wait:</h4>
            <ol className="text-sm text-gray-700 space-y-1">
              <li>1. We're preparing your Telegram channel access</li>
              <li>2. The "Join Telegram Channel" button will be enabled shortly</li>
              <li>3. Please wait for the preparation to complete</li>
            </ol>
          </div>
        )}
        
        {verificationState === 'verifying' && (
          <div className="bg-gray-50 rounded-xl p-4 mb-6">
            <h4 className="font-semibold text-gray-900 mb-2">Instructions:</h4>
            <ol className="text-sm text-gray-700 space-y-1">
              <li>1. Telegram channel opened in a new tab</li>
              <li>2. Click "Join" button on the channel</li>
              <li>3. Wait for verification to complete</li>
              <li>4. You'll automatically proceed to the next step</li>
            </ol>
          </div>
        )}

        {/* Footer Note */}
        <p className="text-xs text-gray-500 text-center mt-4">
          All questions are required to proceed with your free recharge
        </p>
      </div>
    </div>
  );
};