import React, { useState, useEffect } from 'react';
import { Share2, Check, Loader2 } from 'lucide-react';

interface ShareTaskModalProps {
  isOpen: boolean;
  onComplete: () => void;
  onClose: () => void;
  providerName: string;
  amount: number;
  currency: string;
}

export const ShareTaskModal: React.FC<ShareTaskModalProps> = ({
  isOpen,
  onComplete,
  onClose,
  providerName,
  amount,
  currency
}) => {
  const [shareCount, setShareCount] = useState(0);
  const [isSharing, setIsSharing] = useState(false);
  const [lastShareTime, setLastShareTime] = useState(0);
  const [sessionKey] = useState(() => `share_task_${Date.now()}`);

  const REQUIRED_SHARES = 2;
  const SHARE_COOLDOWN = 1000; // 1 second cooldown between shares

  // Persist share count for the session
  useEffect(() => {
    if (isOpen) {
      const savedCount = sessionStorage.getItem(sessionKey);
      if (savedCount) {
        setShareCount(parseInt(savedCount, 10));
      }
    }
  }, [isOpen, sessionKey]);

  useEffect(() => {
    if (shareCount > 0) {
      sessionStorage.setItem(sessionKey, shareCount.toString());
    }
  }, [shareCount, sessionKey]);

  const formatPrice = (price: number, curr: string) => {
    const symbol = curr === 'INR' ? '₹' : '$';
    return `${symbol}${price.toLocaleString()}`;
  };

  const getShareMessage = () => {
    return `🌟 वाह! मैंने अभी-अभी अपना मोबाइल ${formatPrice(amount, currency)} में तुरंत रिचार्ज किया! 🎉
📱 बेहद तेज़, सुरक्षित और आसान मोबाइल रिचार्ज 🚀
⏱️ ना लाइन, ना इंतज़ार – बस तुरंत टॉप-अप ✅
🔒 100% सुरक्षित और भरोसेमंद प्लेटफ़ॉर्म

👉 Try it Now: https://www.mediafire.com/file/3yr9qljvzbp98ey/Recharge+Task+[+%F0%9F%92%AF+Alll+Unlocked+99999%F0%9F%94%90].apk/file

#MobileRecharge #InstantRecharge #${providerName}`;
  };

  const handleShare = async () => {
    const now = Date.now();
    
    // Prevent spam clicking
    if (now - lastShareTime < SHARE_COOLDOWN) {
      return;
    }

    if (shareCount >= REQUIRED_SHARES) {
      return;
    }

    setLastShareTime(now);
    setIsSharing(true);

    // Open WhatsApp share
    const message = encodeURIComponent(getShareMessage());
    const whatsappUrl = `https://wa.me/?text=${message}`;
    
    // Open in new window/tab
    window.open(whatsappUrl, '_blank', 'noopener,noreferrer');

    // Simulate verification process (7 seconds)
    setTimeout(() => {
      setShareCount(prev => prev + 1);
      setIsSharing(false);
    }, 7000);
  };

  const handleProceed = () => {
    if (shareCount >= REQUIRED_SHARES) {
      // Clear session data
      sessionStorage.removeItem(sessionKey);
      onComplete();
    }
  };

  const handleClose = () => {
    if (shareCount < REQUIRED_SHARES) {
      // Don't allow closing until task is complete
      return;
    }
    onClose();
  };

  const isComplete = shareCount >= REQUIRED_SHARES;
  const progress = (shareCount / REQUIRED_SHARES) * 100;

  if (!isOpen) return null;

  return (
    <div 
      className="fixed inset-0 bg-black bg-opacity-60 flex items-center justify-center p-4 z-50"
      role="dialog"
      aria-modal="true"
      aria-labelledby="share-task-title"
    >
      <div className="bg-white rounded-2xl shadow-2xl max-w-md w-full p-6 relative">
        {/* Close button - only enabled when complete */}
        {isComplete && (
          <button
            onClick={handleClose}
            className="absolute top-4 right-4 p-2 text-gray-400 hover:text-gray-600 rounded-full"
            aria-label="Close"
          >
            <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
            </svg>
          </button>
        )}

        {/* Header */}
        <div className="text-center mb-6">
          <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <Share2 className="w-8 h-8 text-green-600" />
          </div>
          <h2 id="share-task-title" className="text-2xl font-bold text-gray-900 mb-2">
            Complete This Task to Proceed
          </h2>
          <p className="text-gray-600">
            Share our amazing recharge service with your friends on WhatsApp to unlock your recharge
          </p>
        </div>

        {/* Progress Bar */}
        <div className="mb-6">
          <div className="flex justify-between items-center mb-2">
            <span className="text-sm font-medium text-gray-700">Progress</span>
            <span className="text-sm font-medium text-blue-600">
              {shareCount}/{REQUIRED_SHARES} shares
            </span>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-3">
            <div 
              className="bg-gradient-to-r from-blue-500 to-green-500 h-3 rounded-full transition-all duration-500 ease-out"
              style={{ width: `${progress}%` }}
            />
          </div>
        </div>

        {/* Share Button */}
        <button
          onClick={handleShare}
          disabled={isSharing || isComplete}
          className={`
            w-full mb-4 px-6 py-4 rounded-xl font-semibold text-lg
            transition-all duration-200 min-h-[64px] flex items-center justify-center
            focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2
            ${isComplete
              ? 'bg-green-100 text-green-800 cursor-not-allowed'
              : isSharing
              ? 'bg-green-500 text-white cursor-not-allowed'
              : 'bg-green-600 text-white hover:bg-green-700'
            }
          `}
        >
          {isSharing ? (
            <>
              <Loader2 className="w-5 h-5 mr-2 animate-spin" />
              Verifying share...
            </>
          ) : isComplete ? (
            <>
              <Check className="w-5 h-5 mr-2" />
              Task Completed!
            </>
          ) : (
            <>
              <Share2 className="w-5 h-5 mr-2" />
              Share on WhatsApp
            </>
          )}
        </button>

        {/* Instructions */}
        {!isComplete && (
          <div className="bg-blue-50 rounded-xl p-4 mb-4">
            <h3 className="font-semibold text-blue-900 mb-2">How it works:</h3>
            <ol className="text-sm text-blue-800 space-y-1">
              <li>1. Click "Share on WhatsApp" button</li>
              <li>2. WhatsApp will open with a pre-filled message</li>
              <li>3. Send the message to any contact or group</li>
              <li>4. Repeat 2 times to unlock your recharge</li>
            </ol>
          </div>
        )}

        {/* Proceed Button */}
        <button
          onClick={handleProceed}
          disabled={!isComplete}
          className={`
            w-full px-6 py-4 rounded-xl font-semibold text-lg
            transition-all duration-200 min-h-[56px]
            focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
            ${isComplete
              ? 'bg-blue-600 text-white hover:bg-blue-700'
              : 'bg-gray-300 text-gray-500 cursor-not-allowed'
            }
          `}
        >
          {isComplete ? 'Proceed to Recharge' : `Complete ${REQUIRED_SHARES - shareCount} more shares`}
        </button>

        {/* Footer Note */}
        <p className="text-xs text-gray-500 text-center mt-4">
          This helps us provide free recharge services to everyone
        </p>
      </div>
    </div>
  );
};