import React, { useState } from 'react';
import { Check, Loader2, X, MessageCircle, Camera, Phone } from 'lucide-react';

interface SocialTask {
  id: string;
  title: string;
  link: string;
  icon: React.ReactNode;
  color: string;
}

interface SocialTasksModalProps {
  isOpen: boolean;
  onComplete: () => void;
  onClose: () => void;
}

export const SocialTasksModal: React.FC<SocialTasksModalProps> = ({
  isOpen,
  onComplete,
  onClose
}) => {
  const [completedTasks, setCompletedTasks] = useState<Set<string>>(new Set());
  const [loadingTasks, setLoadingTasks] = useState<Set<string>>(new Set());

  const socialTasks: SocialTask[] = [
    {
      id: 'telegram1',
      title: 'Join Telegram',
      link: 'tg://join?invite=BnIB1_A-iTM4ZWY1',
      icon: <MessageCircle className="w-5 h-5" />,
      color: 'bg-blue-500 hover:bg-blue-600'
    },
    {
      id: 'telegram2',
      title: 'Join Telegram 2',
      link: 'tg://resolve?domain=editorggr',
      icon: <MessageCircle className="w-5 h-5" />,
      color: 'bg-blue-500 hover:bg-blue-600'
    },
    {
      id: 'instagram',
      title: '2 Comment + Put Story',
      link: 'https://www.instagram.com/reel/DOImT8tkc8p',
      icon: <Camera className="w-5 h-5" />,
      color: 'bg-gradient-to-r from-purple-500 to-orange-500 hover:from-purple-600 hover:to-orange-600'
    },
    {
      id: 'whatsapp',
      title: 'Follow on WhatsApp Channel',
      link: 'https://whatsapp.com/channel/0029Va4vAtxHltY2F0H2Cf05',
      icon: <Phone className="w-5 h-5" />,
      color: 'bg-green-500 hover:bg-green-600'
    }
  ];

  const handleTaskClick = (task: SocialTask) => {
    if (completedTasks.has(task.id) || loadingTasks.has(task.id)) {
      return;
    }

    // Add to loading state
    setLoadingTasks(prev => new Set([...prev, task.id]));

    // Open the link
    window.open(task.link, '_blank', 'noopener,noreferrer');

    // Simulate verification process (5 seconds)
    setTimeout(() => {
      setLoadingTasks(prev => {
        const newSet = new Set(prev);
        newSet.delete(task.id);
        return newSet;
      });
      
      setCompletedTasks(prev => new Set([...prev, task.id]));
    }, 5000);
  };

  const handleProceed = () => {
    if (completedTasks.size === socialTasks.length) {
      onComplete();
    }
  };

  const isAllCompleted = completedTasks.size === socialTasks.length;
  const progress = (completedTasks.size / socialTasks.length) * 100;

  if (!isOpen) return null;

  return (
    <div 
      className="fixed inset-0 bg-black bg-opacity-60 flex items-center justify-center p-4 z-50"
      role="dialog"
      aria-modal="true"
      aria-labelledby="social-tasks-title"
    >
      <div className="bg-white rounded-2xl shadow-2xl max-w-md w-full p-6 relative max-h-[90vh] overflow-y-auto">
        {/* Close button */}
        <button
          onClick={onClose}
          className="absolute top-4 right-4 p-2 text-gray-400 hover:text-gray-600 rounded-full z-10"
          aria-label="Close"
        >
          <X className="w-5 h-5" />
        </button>

        {/* Header */}
        <div className="text-center mb-6">
          <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4 relative">
            <div className="text-2xl">🎉</div>
          </div>
          <h2 id="social-tasks-title" className="text-2xl font-bold text-gray-900 mb-2">
            Almost Done!
          </h2>
          <p className="text-gray-600">
            Complete these tasks to unlock the number
          </p>
        </div>

        {/* Progress Bar */}
        <div className="mb-6">
          <div className="flex justify-between items-center mb-2">
            <span className="text-sm font-medium text-gray-700">Progress</span>
            <span className="text-sm font-medium text-blue-600">
              {completedTasks.size} of {socialTasks.length} tasks completed
            </span>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-3">
            <div 
              className="bg-gradient-to-r from-blue-500 to-green-500 h-3 rounded-full transition-all duration-500 ease-out"
              style={{ width: `${progress}%` }}
            />
          </div>
        </div>

        {/* Social Tasks */}
        <div className="space-y-3 mb-6">
          {socialTasks.map((task) => {
            const isCompleted = completedTasks.has(task.id);
            const isLoading = loadingTasks.has(task.id);

            return (
              <button
                key={task.id}
                onClick={() => handleTaskClick(task)}
                disabled={isCompleted || isLoading}
                className={`
                  w-full px-6 py-4 rounded-xl font-semibold text-white
                  transition-all duration-200 min-h-[56px] flex items-center justify-center
                  focus:outline-none focus:ring-2 focus:ring-offset-2
                  ${isCompleted
                    ? 'bg-green-500 text-white cursor-not-allowed'
                    : isLoading
                    ? 'bg-gray-600 text-white cursor-not-allowed'
                    : `${task.color} text-white`
                  }
                `}
              >
                {isLoading ? (
                  <>
                    <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                    <span className="text-white">Joining...</span>
                  </>
                ) : isCompleted ? (
                  <>
                    <Check className="w-5 h-5 mr-2 text-white" />
                    <span className="text-white">Completed!</span>
                  </>
                ) : (
                  <>
                    {task.icon}
                    <span className="ml-2 text-white">{task.title}</span>
                  </>
                )}
              </button>
            );
          })}
        </div>

        {/* Proceed Button */}
        <button
          onClick={handleProceed}
          disabled={!isAllCompleted}
          className={`
            w-full px-6 py-4 rounded-xl font-semibold text-lg
            transition-all duration-200 min-h-[56px]
            focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
            ${isAllCompleted
              ? 'bg-blue-600 text-white hover:bg-blue-700'
              : 'bg-gray-300 text-gray-500 cursor-not-allowed'
            }
          `}
        >
          {isAllCompleted ? 'Proceed Free Recharge' : `Complete ${socialTasks.length - completedTasks.size} more tasks`}
        </button>

        {/* Footer Note */}
        <p className="text-xs text-gray-500 text-center mt-4">
          This helps us grow our community and provide better services
        </p>
      </div>
    </div>
  );
};