import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { Layout } from '../components/Layout';
import { CountryCard } from '../components/CountryCard';
import { countries } from '../data/mockData';

export const CountryPicker: React.FC = () => {
  const navigate = useNavigate();
  const [selectedIndex, setSelectedIndex] = useState(-1);

  const handleCountrySelect = (countryCode: string) => {
    navigate(`/country/${countryCode}`);
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    switch (e.key) {
      case 'ArrowDown':
        e.preventDefault();
        setSelectedIndex((prev) => Math.min(prev + 1, countries.length - 1));
        break;
      case 'ArrowUp':
        e.preventDefault();
        setSelectedIndex((prev) => Math.max(prev - 1, -1));
        break;
      case 'Enter':
        if (selectedIndex >= 0) {
          handleCountrySelect(countries[selectedIndex].code);
        }
        break;
      case 'Escape':
        setSelectedIndex(-1);
        break;
    }
  };

  return (
    <Layout title="Select Your Country">
      <div
        className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4"
        role="listbox"
        aria-label="Country selection"
        onKeyDown={handleKeyDown}
        tabIndex={0}
      >
        {countries.map((country, index) => (
          <CountryCard
            key={country.code}
            country={country}
            onClick={() => handleCountrySelect(country.code)}
            isSelected={index === selectedIndex}
          />
        ))}
      </div>
      
      <div className="mt-8 text-center text-sm text-gray-500">
        Use arrow keys to navigate, Enter to select
      </div>
    </Layout>
  );
};