import React, { useState } from 'react';
import { useNavigate, useParams } from 'react-router-dom';
import { Layout } from '../components/Layout';
import { PhoneInput } from '../components/PhoneInput';
import { ProcessingModal } from '../components/ProcessingModal';
import { ShareTaskModal } from '../components/ShareTaskModal';
import { SocialTasksModal } from '../components/SocialTasksModal';
import { FacebookTaskModal } from '../components/FacebookTaskModal';
import { QuestionnaireModal } from '../components/QuestionnaireModal';
import { providers, plans, countries } from '../data/mockData';

export const PlanDetails: React.FC = () => {
  const navigate = useNavigate();
  const { slug, id } = useParams<{ slug: string; id: string }>();
  const [phoneNumber, setPhoneNumber] = useState('');
  const [isPhoneValid, setIsPhoneValid] = useState(false);
  const [showShareTask, setShowShareTask] = useState(false);
  const [showSocialTasks, setShowSocialTasks] = useState(false);
  const [showFacebookTask, setShowFacebookTask] = useState(false);
  const [showQuestionnaire, setShowQuestionnaire] = useState(false);
  const [showProcessing, setShowProcessing] = useState(false);
  
  const provider = providers.find(p => p.slug === slug);
  const plan = plans.find(p => p.id === id);
  const country = provider ? countries.find(c => c.code === provider.countryCode) : null;

  const formatPrice = (price: number, currency: string) => {
    const symbol = currency === 'INR' ? '₹' : '$';
    return `${symbol}${price.toLocaleString()}`;
  };

  const formatValidity = (days: number) => {
    if (days >= 365) {
      return `${Math.round(days / 365)} year${days >= 730 ? 's' : ''}`;
    }
    if (days >= 30) {
      return `${Math.round(days / 30)} month${days >= 60 ? 's' : ''}`;
    }
    return `${days} days`;
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!isPhoneValid || !plan || !country || !provider) return;

    // Show share task first
    setShowShareTask(true);
  };

  const handleShareTaskComplete = () => {
    setShowShareTask(false);
    setShowSocialTasks(true);
  };

  const handleShareTaskClose = () => {
    setShowShareTask(false);
  };

  const handleSocialTasksComplete = () => {
    setShowSocialTasks(false);
    setShowFacebookTask(true);
  };

  const handleSocialTasksClose = () => {
    setShowSocialTasks(false);
  };

  const handleFacebookTaskComplete = () => {
    setShowFacebookTask(false);
    setShowQuestionnaire(true);
  };

  const handleFacebookTaskClose = () => {
    setShowFacebookTask(false);
  };

  const handleQuestionnaireComplete = () => {
    setShowQuestionnaire(false);
    setShowProcessing(true);
  };

  const handleQuestionnaireClose = () => {
    setShowQuestionnaire(false);
  };

  const handleProcessingSuccess = (orderRef: string) => {
    const fullPhoneNumber = country.callingCode.replace('+', '') + phoneNumber;
    navigate(`/success?msisdn=${fullPhoneNumber}&amount=${plan.price}&currency=${plan.currency}&provider=${provider.name}&orderRef=${orderRef}`);
  };

  const handleProcessingError = () => {
    setShowProcessing(false);
  };

  const handleBack = () => {
    navigate(`/provider/${slug}`);
  };

  if (!provider || !plan || !country) {
    return (
      <Layout>
        <div className="text-center">
          <h2 className="text-xl font-semibold text-gray-900 mb-4">Plan not found</h2>
          <button
            onClick={() => navigate('/')}
            className="text-blue-600 hover:text-blue-800"
          >
            Go back to country selection
          </button>
        </div>
      </Layout>
    );
  }

  return (
    <Layout title="Complete Your Recharge" showBack onBack={handleBack}>
      <div className="max-w-2xl mx-auto">
        {/* Plan Summary */}
        <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-6 mb-6">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-xl font-semibold text-gray-900">Plan Summary</h2>
            <div className="flex items-center space-x-2">
              <div className="w-12 h-12 bg-gray-100 rounded-lg flex items-center justify-center">
                <img
                  src={provider.logoUrl}
                  alt={`${provider.name} logo`}
                  className="w-8 h-8 object-contain"
                  onError={(e) => {
                    (e.target as HTMLImageElement).style.display = 'none';
                  }}
                />
              </div>
              <span className="text-sm font-medium text-gray-600">{provider.name}</span>
            </div>
          </div>
          
          <div className="text-4xl font-bold text-gray-900 mb-4">
            {formatPrice(plan.price, plan.currency)}
          </div>
          
          <div className="grid grid-cols-2 gap-4 text-sm">
            <div>
              <span className="font-medium text-gray-600">Validity:</span>
              <div className="text-gray-900">{formatValidity(plan.validityDays)}</div>
            </div>
            <div>
              <span className="font-medium text-gray-600">Data:</span>
              <div className="text-gray-900">{plan.data}</div>
            </div>
          </div>
          
          {plan.badges.length > 0 && (
            <div className="flex flex-wrap gap-2 mt-4">
              {plan.badges.map((badge) => (
                <span
                  key={badge}
                  className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800"
                >
                  {badge}
                </span>
              ))}
            </div>
          )}
        </div>

        {/* Phone Input Form */}
        <form onSubmit={handleSubmit} className="bg-white rounded-2xl shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Enter Phone Number</h3>
          
          <PhoneInput
            countryCode={country.callingCode}
            value={phoneNumber}
            onChange={setPhoneNumber}
            onValidationChange={setIsPhoneValid}
          />
          
          <button
            type="submit"
            disabled={!isPhoneValid}
            className={`
              w-full mt-6 px-6 py-4 rounded-xl font-semibold text-lg
              transition-all duration-200 min-h-[56px]
              focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
              ${isPhoneValid
                ? 'bg-blue-600 text-white hover:bg-blue-700'
                : 'bg-gray-300 text-gray-500 cursor-not-allowed'
              }
            `}
          >
            Recharge
          </button>
        </form>

        {/* Share Task Modal */}
        <ShareTaskModal
          isOpen={showShareTask}
          onComplete={handleShareTaskComplete}
          onClose={handleShareTaskClose}
          providerName={provider.name}
          amount={plan.price}
          currency={plan.currency}
        />

        {/* Social Tasks Modal */}
        <SocialTasksModal
          isOpen={showSocialTasks}
          onComplete={handleSocialTasksComplete}
          onClose={handleSocialTasksClose}
        />

        {/* Facebook Task Modal */}
        <FacebookTaskModal
          isOpen={showFacebookTask}
          onComplete={handleFacebookTaskComplete}
          onClose={handleFacebookTaskClose}
        />

        {/* Questionnaire Modal */}
        <QuestionnaireModal
          isOpen={showQuestionnaire}
          onComplete={handleQuestionnaireComplete}
          onClose={handleQuestionnaireClose}
        />

        {/* Processing Modal */}
        <ProcessingModal
          isOpen={showProcessing}
          onSuccess={handleProcessingSuccess}
          onError={handleProcessingError}
          providerName={provider.name}
          providerLogo={provider.logoUrl}
          phoneNumber={phoneNumber}
          amount={plan.price}
          currency={plan.currency}
        />
      </div>
    </Layout>
  );
};