import React, { useState } from 'react';
import { useNavigate, useParams } from 'react-router-dom';
import { Layout } from '../components/Layout';
import { PlanCard } from '../components/PlanCard';
import { providers, plans } from '../data/mockData';

export const PlansListing: React.FC = () => {
  const navigate = useNavigate();
  const { slug } = useParams<{ slug: string }>();
  const [searchTerm, setSearchTerm] = useState('');
  
  const provider = providers.find(p => p.slug === slug);
  
  const handlePlanSelect = (planId: string) => {
    navigate(`/provider/${slug}/plan/${planId}`);
  };

  const handleBack = () => {
    if (provider) {
      navigate(`/country/${provider.countryCode}`);
    } else {
      navigate('/');
    }
  };

  const filteredPlans = plans.filter(plan =>
    plan.data.toLowerCase().includes(searchTerm.toLowerCase()) ||
    plan.price.toString().includes(searchTerm)
  );

  // Sort plans by best value (price per day)
  const sortedPlans = [...filteredPlans].sort((a, b) => {
    const aValuePerDay = a.price / a.validityDays;
    const bValuePerDay = b.price / b.validityDays;
    return aValuePerDay - bValuePerDay;
  });

  if (!provider) {
    return (
      <Layout>
        <div className="text-center">
          <h2 className="text-xl font-semibold text-gray-900 mb-4">Provider not found</h2>
          <button
            onClick={() => navigate('/')}
            className="text-blue-600 hover:text-blue-800"
          >
            Go back to country selection
          </button>
        </div>
      </Layout>
    );
  }

  return (
    <Layout title={`${provider.name} Plans`} showBack onBack={handleBack}>
      <div className="mb-6">
        <label htmlFor="search" className="block text-sm font-medium text-gray-700 mb-2">
          Search plans
        </label>
        <input
          id="search"
          type="text"
          value={searchTerm}
          onChange={(e) => setSearchTerm(e.target.value)}
          placeholder="Search by data or price..."
          className="
            w-full px-4 py-3 border border-gray-300 rounded-xl
            focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500
          "
        />
      </div>

      {sortedPlans.length === 0 ? (
        <div className="text-center py-12">
          <h3 className="text-lg font-medium text-gray-900 mb-2">No plans found</h3>
          <p className="text-gray-600">
            {searchTerm 
              ? 'Try adjusting your search terms'
              : 'No plans are available for this provider'
            }
          </p>
        </div>
      ) : (
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {sortedPlans.map((plan) => (
            <PlanCard
              key={plan.id}
              plan={plan}
              onClick={() => handlePlanSelect(plan.id)}
            />
          ))}
        </div>
      )}
    </Layout>
  );
};