import React, { useEffect } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { Layout } from '../components/Layout';
import { ConfettiEffect } from '../components/ConfettiEffect';
import { providers } from '../data/mockData';

export const Success: React.FC = () => {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  
  const msisdn = searchParams.get('msisdn');
  const amount = searchParams.get('amount');
  const currency = searchParams.get('currency') || 'INR';
  const provider = searchParams.get('provider');
  const orderRef = searchParams.get('orderRef');

  // Find provider details
  const providerDetails = providers.find((p: any) => p.name === provider);

  const formatPhoneNumber = (number: string) => {
    if (number.startsWith('91') && number.length === 12) {
      // Indian number: +91 XXXXX XXXXX
      const withoutCode = number.substring(2);
      return `+91 ${withoutCode.substring(0, 5)} ${withoutCode.substring(5)}`;
    }
    return `+${number}`;
  };

  const formatPrice = (price: string, curr: string) => {
    const symbol = curr === 'INR' ? '₹' : '$';
    return `${symbol}${parseInt(price).toLocaleString()}`;
  };

  useEffect(() => {
    // Announce success to screen readers
    const announcement = document.createElement('div');
    announcement.setAttribute('role', 'status');
    announcement.setAttribute('aria-live', 'polite');
    announcement.className = 'sr-only';
    announcement.textContent = 'Recharge completed successfully';
    document.body.appendChild(announcement);

    return () => {
      document.body.removeChild(announcement);
    };
  }, []);

  if (!msisdn || !amount) {
    return (
      <Layout>
        <div className="text-center">
          <h2 className="text-xl font-semibold text-gray-900 mb-4">Invalid request</h2>
          <button
            onClick={() => navigate('/')}
            className="text-blue-600 hover:text-blue-800"
          >
            Go back to home
          </button>
        </div>
      </Layout>
    );
  }

  return (
    <Layout>
      <ConfettiEffect trigger={true} duration={3000} />
      <div className="max-w-2xl mx-auto text-center">
        <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-8">
          {/* Success Hero */}
          <div className="mb-8">
            {/* Provider Branding */}
            {providerDetails && (
              <div className="flex items-center justify-center mb-6">
                <div className="w-16 h-16 rounded-full overflow-hidden flex items-center justify-center mr-3">
                  <img
                    src={providerDetails.logoUrl}
                    alt={`${providerDetails.name} logo`}
                    className="w-full h-full object-cover"
                    onError={(e) => {
                      (e.target as HTMLImageElement).style.display = 'none';
                    }}
                  />
                </div>
                <div className="text-left">
                  <div className="text-2xl font-bold text-gray-900">{providerDetails.name}</div>
                  <div className="text-sm text-gray-500">{providerDetails.tagline}</div>
                </div>
              </div>
            )}
            
            <h1 className="text-3xl sm:text-4xl lg:text-5xl font-bold text-gray-900 mb-4 break-words">
              Congratulations 🎉
            </h1>
            
            <div className="text-lg sm:text-xl lg:text-2xl text-gray-800 mb-6 break-words" lang="hi">
              {formatPhoneNumber(msisdn)} पर {formatPrice(amount, currency)} सफलतापूर्वक रिचार्ज कर दिया गया है।
            </div>
            
            <div className="text-base sm:text-lg text-gray-600 break-words">
              Your {provider} recharge of {formatPrice(amount, currency)} for {formatPhoneNumber(msisdn)} has been completed successfully.
            </div>
          </div>

          {/* Order Details */}
          <div className="bg-gray-50 rounded-xl p-6 mb-8">
            <div className="text-sm text-gray-600 mb-2">Order Reference</div>
            <div className="text-lg sm:text-xl font-mono font-semibold text-gray-900 break-all">
              {orderRef || 'REF' + Math.random().toString(36).substr(2, 9).toUpperCase()}
            </div>
          </div>

          {/* Actions */}
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <button
              onClick={() => navigate('/')}
              className="
                px-6 py-3 bg-blue-600 text-white font-semibold rounded-xl
                hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
                transition-colors duration-200
              "
            >
              Back to Plans
            </button>
            <button
              onClick={() => window.print()}
              className="
                px-6 py-3 bg-gray-200 text-gray-800 font-semibold rounded-xl
                hover:bg-gray-300 focus:outline-none focus:ring-2 focus:ring-gray-500 focus:ring-offset-2
                transition-colors duration-200
              "
            >
              Print Receipt
            </button>
          </div>
        </div>
      </div>
    </Layout>
  );
};