// Free Fire Tournament Website JavaScript

document.addEventListener('DOMContentLoaded', function() {
    console.log('Free Fire Tournament Website Loading...');

    // Mobile Navigation Toggle
    const navToggle = document.getElementById('nav-toggle');
    const navMenu = document.getElementById('nav-menu');
    const navLinks = document.querySelectorAll('.nav-link');

    // Ensure mobile toggle works
    if (navToggle && navMenu) {
        navToggle.addEventListener('click', function(e) {
            e.preventDefault();
            console.log('Nav toggle clicked');
            navMenu.classList.toggle('active');
            navToggle.classList.toggle('active');
        });
    }

    // Close mobile menu when clicking on a link
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            // Don't prevent default here, let the smooth scroll work
            navMenu.classList.remove('active');
            navToggle.classList.remove('active');
        });
    });

    // Smooth scrolling for navigation links
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const targetId = this.getAttribute('href');
            console.log('Navigating to:', targetId);
            
            if (targetId && targetId.startsWith('#')) {
                const targetSection = document.querySelector(targetId);
                
                if (targetSection) {
                    const headerHeight = document.querySelector('.header').offsetHeight || 80;
                    const targetPosition = targetSection.offsetTop - headerHeight;
                    
                    window.scrollTo({
                        top: targetPosition,
                        behavior: 'smooth'
                    });
                    console.log('Scrolling to:', targetPosition);
                } else {
                    console.log('Target section not found:', targetId);
                }
            }
        });
    });

    // Hero button smooth scrolling
    const heroBtns = document.querySelectorAll('.hero-actions .btn');
    heroBtns.forEach(btn => {
        btn.addEventListener('click', function(e) {
            const href = this.getAttribute('href');
            if (href && href.startsWith('#')) {
                e.preventDefault();
                const targetSection = document.querySelector(href);
                if (targetSection) {
                    const headerHeight = document.querySelector('.header').offsetHeight || 80;
                    const targetPosition = targetSection.offsetTop - headerHeight;
                    
                    window.scrollTo({
                        top: targetPosition,
                        behavior: 'smooth'
                    });
                }
            }
        });
    });

    // Header scroll effect
    const header = document.querySelector('.header');
    let lastScrollY = window.scrollY;

    window.addEventListener('scroll', function() {
        const currentScrollY = window.scrollY;
        
        if (currentScrollY > 100) {
            header.style.background = 'rgba(10, 14, 26, 0.98)';
            header.style.boxShadow = '0 2px 20px rgba(255, 71, 87, 0.1)';
        } else {
            header.style.background = 'rgba(10, 14, 26, 0.95)';
            header.style.boxShadow = 'none';
        }
        
        lastScrollY = currentScrollY;
    });

    // Registration Form Handling
    const registrationForm = document.getElementById('registration-form');
    
    if (registrationForm) {
        // Ensure all form inputs are properly enabled
        const formInputs = registrationForm.querySelectorAll('input, textarea, select');
        formInputs.forEach(input => {
            input.removeAttribute('disabled');
            input.removeAttribute('readonly');
        });

        registrationForm.addEventListener('submit', function(e) {
            e.preventDefault();
            console.log('Registration form submitted');
            
            // Get form data
            const formData = new FormData(this);
            const teamData = {};
            
            // Extract all form data
            for (let [key, value] of formData.entries()) {
                teamData[key] = value;
            }
            
            console.log('Form data:', teamData);

            // Validate form data
            if (validateRegistrationForm(teamData)) {
                // Show success message
                showNotification('Registration submitted successfully! We will contact you soon.', 'success');
                
                // Reset form
                this.reset();
                
                // Scroll to top
                window.scrollTo({ top: 0, behavior: 'smooth' });
            } else {
                showNotification('Please fill in all required fields correctly.', 'error');
            }
        });
    }

    // Contact Form Handling
    const contactForm = document.getElementById('contact-form');
    
    if (contactForm) {
        contactForm.addEventListener('submit', function(e) {
            e.preventDefault();
            console.log('Contact form submitted');
            
            const formInputs = this.querySelectorAll('input, textarea');
            let isValid = true;
            
            formInputs.forEach(input => {
                if (!input.value.trim()) {
                    isValid = false;
                    input.style.borderColor = '#ff4757';
                } else {
                    input.style.borderColor = '';
                }
            });
            
            if (isValid) {
                showNotification('Message sent successfully! We will get back to you soon.', 'success');
                this.reset();
            } else {
                showNotification('Please fill in all fields.', 'error');
            }
        });
    }

    // Form validation functions
    function validateRegistrationForm(data) {
        console.log('Validating form data:', data);
        
        // Check if all required fields are filled
        const requiredFields = ['teamName', 'captainName', 'captainContact', 'player1', 'player2', 'player3', 'player4', 'email'];
        
        for (let field of requiredFields) {
            if (!data[field] || data[field].toString().trim() === '') {
                console.log('Missing field:', field);
                return false;
            }
        }
        
        // Validate email format
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(data.email)) {
            console.log('Invalid email format');
            return false;
        }
        
        return true;
    }

    // Notification system
    function showNotification(message, type = 'info') {
        console.log('Showing notification:', message, type);
        
        // Remove existing notifications
        const existingNotifications = document.querySelectorAll('.notification');
        existingNotifications.forEach(notification => notification.remove());
        
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        
        // Set notification styles and content
        const bgColor = type === 'success' ? '#27ae60' : type === 'error' ? '#e74c3c' : '#3498db';
        
        notification.innerHTML = `
            <div style="display: flex; justify-content: space-between; align-items: center; gap: 1rem;">
                <span>${message}</span>
                <button onclick="this.parentElement.parentElement.remove()" style="background: none; border: none; color: white; font-size: 1.5rem; cursor: pointer; padding: 0; width: 20px; height: 20px;">&times;</button>
            </div>
        `;
        
        notification.style.cssText = `
            position: fixed;
            top: 100px;
            right: 20px;
            background: ${bgColor};
            color: white;
            padding: 1rem 1.5rem;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            z-index: 10000;
            max-width: 400px;
            font-weight: 500;
            animation: slideInFromRight 0.3s ease;
        `;
        
        // Add animation
        if (!document.querySelector('#notification-animation-styles')) {
            const animationStyle = document.createElement('style');
            animationStyle.id = 'notification-animation-styles';
            animationStyle.textContent = `
                @keyframes slideInFromRight {
                    from {
                        transform: translateX(100%);
                        opacity: 0;
                    }
                    to {
                        transform: translateX(0);
                        opacity: 1;
                    }
                }
            `;
            document.head.appendChild(animationStyle);
        }
        
        // Add to document
        document.body.appendChild(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.style.animation = 'slideInFromRight 0.3s ease reverse';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.remove();
                    }
                }, 300);
            }
        }, 5000);
    }

    // Real-time form validation
    const allFormInputs = document.querySelectorAll('.form-control');
    allFormInputs.forEach(input => {
        // Remove any potential blocking attributes
        input.removeAttribute('disabled');
        input.removeAttribute('readonly');
        
        input.addEventListener('blur', function() {
            if (this.hasAttribute('required') && !this.value.trim()) {
                this.style.borderColor = '#ff4757';
                this.style.boxShadow = '0 0 5px rgba(255, 71, 87, 0.3)';
            } else {
                this.style.borderColor = '';
                this.style.boxShadow = '';
            }
        });
        
        input.addEventListener('input', function() {
            if (this.style.borderColor === 'rgb(255, 71, 87)') {
                this.style.borderColor = '';
                this.style.boxShadow = '';
            }
        });

        // Test input functionality
        input.addEventListener('focus', function() {
            console.log('Input focused:', this.name || this.placeholder);
        });
    });

    // Intersection Observer for animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    }, observerOptions);

    // Add fade-in animation to sections
    const sections = document.querySelectorAll('section');
    sections.forEach(section => {
        section.style.opacity = '0';
        section.style.transform = 'translateY(30px)';
        section.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
        observer.observe(section);
    });

    // Add countdown timer for registration deadline
    function addCountdownTimer() {
        const deadlineDate = new Date('September 25, 2025 23:59:59').getTime();
        const timerElement = document.createElement('div');
        timerElement.className = 'countdown-timer';
        timerElement.style.cssText = `
            position: fixed;
            bottom: 20px;
            right: 20px;
            background: linear-gradient(135deg, #ff4757, #ff6b3d);
            color: white;
            padding: 1rem;
            border-radius: 10px;
            font-family: 'Orbitron', monospace;
            font-weight: 600;
            box-shadow: 0 4px 20px rgba(255, 71, 87, 0.3);
            z-index: 1000;
            text-align: center;
            min-width: 200px;
        `;
        
        const updateTimer = () => {
            const now = new Date().getTime();
            const distance = deadlineDate - now;
            
            if (distance > 0) {
                const days = Math.floor(distance / (1000 * 60 * 60 * 24));
                const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
                const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
                const seconds = Math.floor((distance % (1000 * 60)) / 1000);
                
                timerElement.innerHTML = `
                    <div style="font-size: 0.8rem; margin-bottom: 0.5rem;">REGISTRATION ENDS IN</div>
                    <div style="font-size: 1rem;">${days}d ${hours}h ${minutes}m ${seconds}s</div>
                    <button onclick="this.parentElement.remove()" style="position: absolute; top: 5px; right: 5px; background: none; border: none; color: white; font-size: 1.2rem; cursor: pointer; width: 20px; height: 20px;">&times;</button>
                `;
            } else {
                timerElement.innerHTML = `
                    <div>Registration Closed</div>
                    <button onclick="this.parentElement.remove()" style="position: absolute; top: 5px; right: 5px; background: none; border: none; color: white; font-size: 1.2rem; cursor: pointer; width: 20px; height: 20px;">&times;</button>
                `;
                timerElement.style.background = '#95a5a6';
            }
        };
        
        updateTimer();
        setInterval(updateTimer, 1000);
        
        document.body.appendChild(timerElement);
    }

    // Initialize countdown timer
    addCountdownTimer();

    // Add scroll-to-top button
    const scrollToTopBtn = document.createElement('button');
    scrollToTopBtn.innerHTML = '↑';
    scrollToTopBtn.className = 'scroll-to-top';
    scrollToTopBtn.style.cssText = `
        position: fixed;
        bottom: 260px;
        right: 20px;
        width: 50px;
        height: 50px;
        background: linear-gradient(135deg, #ff4757, #ff6b3d);
        color: white;
        border: none;
        border-radius: 50%;
        font-size: 1.5rem;
        cursor: pointer;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s ease;
        box-shadow: 0 4px 20px rgba(255, 71, 87, 0.3);
    `;
    
    document.body.appendChild(scrollToTopBtn);
    
    // Show/hide scroll to top button
    window.addEventListener('scroll', function() {
        if (window.scrollY > 300) {
            scrollToTopBtn.style.opacity = '1';
            scrollToTopBtn.style.visibility = 'visible';
        } else {
            scrollToTopBtn.style.opacity = '0';
            scrollToTopBtn.style.visibility = 'hidden';
        }
    });
    
    // Scroll to top functionality
    scrollToTopBtn.addEventListener('click', function() {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });

    // Test all interactive elements
    setTimeout(() => {
        console.log('Testing interactive elements...');
        
        // Test navigation links
        const navLinkCount = navLinks.length;
        console.log(`Found ${navLinkCount} navigation links`);
        
        // Test form inputs
        const inputCount = allFormInputs.length;
        console.log(`Found ${inputCount} form inputs`);
        
        // Test mobile toggle
        if (navToggle) {
            console.log('Mobile navigation toggle found');
        } else {
            console.log('Mobile navigation toggle NOT found');
        }
        
        console.log('Free Fire Tournament Website Loaded Successfully! 🔥');
    }, 1000);
});